<?php

function enqueue_cart_scripts() {
    wp_enqueue_script('cart', get_template_directory_uri() . '/assets/js/cart.js');
    
    wp_localize_script('cart-js', 'ajax_object', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'remove_coupon_nonce' => wp_create_nonce('remove_coupon_nonce'),
        'update_cart_nonce' => wp_create_nonce('update_cart_nonce'),
        'add_to_cart_nonce' => wp_create_nonce('add_to_cart_nonce')
    ));
}
add_action('wp_enqueue_scripts', 'enqueue_cart_scripts');

add_action('wp_ajax_add_to_cart', 'handle_add_to_cart');
add_action('wp_ajax_nopriv_add_to_cart', 'handle_add_to_cart');  // For non-logged in users

function handle_add_to_cart() {
    $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
    $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;

    if ($product_id > 0) {
        WC()->cart->add_to_cart($product_id, $quantity);

        wp_send_json_success();
    } else {
        wp_send_json_error();
    }
}

add_action('wp_ajax_get_cart_contents', 'handle_get_cart_contents');
add_action('wp_ajax_nopriv_get_cart_contents', 'handle_get_cart_contents');  // Non-logged-in users

function handle_get_cart_contents() {
    ob_start();
    include(get_template_directory() . '/components/cart-modal.php'); // Dosya yolunu doğru yazdığınızdan emin olun
    $cart_html = ob_get_clean();
    wp_send_json_success(['cart_html' => $cart_html]);
}

add_action('wp_ajax_remove_from_cart', 'remove_from_cart');
add_action('wp_ajax_nopriv_remove_from_cart', 'remove_from_cart');

function remove_from_cart() {
    // AJAX isteğinden ürün ID'sini al
    if (isset($_POST['item_id'])) {
        $item_id = absint($_POST['item_id']); // Güvenlik için ID'yi tam sayıya dönüştür

        // WooCommerce sepeti üzerinden ürün kaldır
        $cart = WC()->cart;
        $cart_items = $cart->get_cart(); // Sepetteki tüm ürünleri al

        foreach ($cart_items as $cart_item_key => $cart_item) {
            if ($cart_item['product_id'] == $item_id) {
                $cart->remove_cart_item($cart_item_key); // Ürünü sepetten kaldır
                wp_send_json_success(); // Başarıyla işlem tamamlandı
                return;
            }
        }
    }

    // Eğer ürün bulunamazsa, başarısız yanıt gönder
    wp_send_json_error();
}


add_action('wp_ajax_add_to_favorites', 'add_to_favorites');
add_action('wp_ajax_nopriv_add_to_favorites', 'add_to_favorites');

function add_to_favorites() {
    // AJAX isteğinden ürün ID'sini al
    if (isset($_POST['item_id'])) {
        $item_id = absint($_POST['item_id']); // Güvenlik için ID'yi tam sayıya dönüştür

        // Favorilere ekleme işlemi
        $user_id = get_current_user_id();  // Kullanıcı ID'sini al
        if ($user_id > 0) {
            // Kullanıcının favorilerine ürün ekle
            $wishlist = get_user_meta($user_id, '_user_favorites', true);
            if (!$wishlist) {
                $wishlist = [];
            }
            if (!in_array($item_id, $wishlist)) {
                $wishlist[] = $item_id;  // Ürünü favorilere ekle
                update_user_meta($user_id, '_user_favorites', $wishlist);
            }
        }

        wp_send_json_success(); // Başarıyla işlem tamamlandı
    }

    wp_send_json_error(); // Hata durumu
}

add_action('wp_ajax_remove_from_favorites', 'remove_from_favorites');
add_action('wp_ajax_nopriv_remove_from_favorites', 'remove_from_favorites');

function remove_from_favorites() {
    // AJAX isteğinden ürün ID'sini al
    if (isset($_POST['item_id'])) {
        $item_id = absint($_POST['item_id']); // Güvenlik için ID'yi tam sayıya dönüştür

        // Favorilerden çıkarma işlemi
        $user_id = get_current_user_id();  // Kullanıcı ID'sini al
        if ($user_id > 0) {
            // Kullanıcının favorilerinden ürün çıkar
            $wishlist = get_user_meta($user_id, '_user_favorites', true);
            if (($key = array_search($item_id, $wishlist)) !== false) {
                unset($wishlist[$key]);  // Ürünü favorilerden çıkar
                update_user_meta($user_id, '_user_favorites', array_values($wishlist));
            }
        }

        wp_send_json_success(); // Başarıyla işlem tamamlandı
    }

    wp_send_json_error(); // Hata durumu
}


add_action( 'wp_ajax_update_cart_item_quantity', 'update_cart_item_quantity' );
add_action( 'wp_ajax_nopriv_update_cart_item_quantity', 'update_cart_item_quantity' );

function update_cart_item_quantity() {
    $product_id = intval($_POST['product_id']);
    $quantity = intval($_POST['quantity']);

    // Sepetteki öğeleri al
    $cart_items = WC()->cart->get_cart();

    foreach ( $cart_items as $item_key => $item ) {
        if ($item['product_id'] == $product_id) {
            // Sepetteki ürünün miktarını güncelle
            WC()->cart->set_quantity($item_key, $quantity);
            break;
        }
    }

    // Sepet güncellenmiş mi kontrol et
    WC()->cart->calculate_totals();
    wp_send_json_success();
}


// AJAX nonce'unu scripts'e ekle
function add_ajax_nonce() {
    wp_localize_script('cart', 'ajax_object', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('apply_coupon_nonce')
    ));
}
add_action('wp_enqueue_scripts', 'add_ajax_nonce');

// Kupon AJAX işleyicisi
add_action('wp_ajax_apply_coupon', 'handle_apply_coupon');
add_action('wp_ajax_nopriv_apply_coupon', 'handle_apply_coupon');

function handle_apply_coupon() {
    // Nonce kontrolü
    check_ajax_referer('apply_coupon_nonce', 'security');

    if (!isset($_POST['coupon_code'])) {
        wp_send_json_error(array('message' => 'Kupon kodu gerekli.'));
    }

    $coupon_code = sanitize_text_field($_POST['coupon_code']);

    if (!WC()->cart) {
        wp_send_json_error(array('message' => 'Sepet başlatılamadı.'));
        return;
    }

    // Kuponu uygula
    $result = WC()->cart->apply_coupon($coupon_code);

    if (wc_get_notices('error')) {
        $notices = wc_get_notices('error');
        $error_message = !empty($notices) ? $notices[0]['notice'] : 'Kupon uygulanamadı.';
        wc_clear_notices(); // Hata mesajlarını temizle
        wp_send_json_error(array('message' => $error_message));
    } else {
        wp_send_json_success(array(
            'message' => 'Kupon başarıyla uygulandı.',
            'cart_total' => WC()->cart->get_cart_total()
        ));
    }

    wp_die();
}

// Kupon kaldırma işlemi için AJAX handler
add_action('wp_ajax_remove_coupon', 'handle_remove_coupon');
add_action('wp_ajax_nopriv_remove_coupon', 'handle_remove_coupon');

function handle_remove_coupon() {

    $coupon_code = isset($_POST['coupon_code']) ? sanitize_text_field($_POST['coupon_code']) : '';
    
    if (empty($coupon_code)) {
        wp_send_json_error(array(
            'message' => 'Kupon kodu bulunamadı.'
        ));
        die();
    }

    // WooCommerce sepetinin varlığını kontrol edelim
    if (!function_exists('WC') || !WC()->cart) {
        wp_send_json_error(array(
            'message' => 'Sepet bulunamadı.'
        ));
        die();
    }

    $removed = WC()->cart->remove_coupon($coupon_code);
    
    if ($removed) {
        WC()->cart->calculate_totals();
        wp_send_json_success(array(
            'message' => 'Kupon başarıyla kaldırıldı',
            'cart_total' => WC()->cart->get_cart_total()
        ));
    } else {
        wp_send_json_error(array(
            'message' => 'Kupon kaldırılamadı.'
        ));
    }

    die();
}

// Sepet boşaltıldığında kuponları temizle
function clear_coupons_on_empty_cart() {
    // wp_loaded hook'undan sonra çalıştığından emin ol
    if (!did_action('wp_loaded')) {
        return;
    }
    
    if (WC()->cart && WC()->cart->is_empty()) {
        WC()->cart->remove_coupons();
        // Sepet zaten boş olduğu için calculate_totals çağrılmasına gerek yok
    }
}
add_action('woocommerce_cart_emptied', 'clear_coupons_on_empty_cart', 20);

// Sepetten ürün kaldırıldığında sepet boş ise kuponları temizle
function check_cart_empty_and_clear_coupons() {
    // wp_loaded hook'undan sonra çalıştığından emin ol
    if (!did_action('wp_loaded')) {
        return;
    }
    
    if (WC()->cart && WC()->cart->is_empty()) {
        WC()->cart->remove_coupons();
        // Sepet zaten boş olduğu için calculate_totals çağrılmasına gerek yok
    }
}
add_action('woocommerce_cart_item_removed', 'check_cart_empty_and_clear_coupons', 20);