<?php
// functions.php dosyasına eklenecek kodlar

// Özel yorum formunu oluştur
function custom_review_form_fields() {
    add_action('comment_form_logged_in_after', 'custom_comment_fields');
    add_action('comment_form_after_fields', 'custom_comment_fields');
}
add_action('init', 'custom_review_form_fields');

// Özel alanları ekle
function custom_comment_fields() {
    ?>
    <p class="comment-form-title">
        <label for="comment_title">Yorum Başlığı <span class="required">*</span></label>
        <input type="text" id="comment_title" name="comment_title" required />
    </p>

    <p class="comment-form-rating">
        <label for="rating">Değerlendirme <span class="required">*</span></label>
        <select name="rating" id="rating" required>
            <option value="">Puan seçin...</option>
            <option value="5">5 yıldız</option>
            <option value="4">4 yıldız</option>
            <option value="3">3 yıldız</option>
            <option value="2">2 yıldız</option>
            <option value="1">1 yıldız</option>
        </select>
    </p>
    <?php
}

// Yorum kaydedilirken özel alanları da kaydet
function save_custom_comment_fields($comment_id) {
    if (isset($_POST['comment_title'])) {
        update_comment_meta($comment_id, 'comment_title', sanitize_text_field($_POST['comment_title']));
    }

    if (isset($_POST['rating'])) {
        update_comment_meta($comment_id, 'rating', intval($_POST['rating']));
    }
}
add_action('comment_post', 'save_custom_comment_fields');

// Form validasyonu
function verify_custom_comment_fields($commentdata) {
    if (!isset($_POST['comment_title']) || empty($_POST['comment_title'])) {
        wp_die('Lütfen bir yorum başlığı girin.');
    }

    if (!isset($_POST['rating']) || empty($_POST['rating'])) {
        wp_die('Lütfen bir değerlendirme puanı seçin.');
    }

    return $commentdata;
}
add_filter('preprocess_comment', 'verify_custom_comment_fields');

// Yorumları görüntüleme fonksiyonu
function display_custom_review($comment) {
    $rating = get_comment_meta($comment->comment_ID, 'rating', true);
    $title = get_comment_meta($comment->comment_ID, 'comment_title', true);
    ?>
    <div class="review-item">
        <div class="review-title"><?php echo esc_html($title); ?></div>
        <div class="review-rating">
            <?php
            for($i = 1; $i <= 5; $i++) {
                echo $i <= $rating ? '★' : '☆';
            }
            ?>
        </div>
        <div class="review-author">
            <?php echo get_comment_author($comment); ?> -
            <?php echo get_comment_date('d.m.Y', $comment); ?>
        </div>
        <div class="review-content">
            <?php echo wpautop(get_comment_text($comment)); ?>
        </div>
    </div>
    <?php
}

// Yorum listesini özelleştir
function custom_review_list($comment, $args, $depth) {
    display_custom_review($comment);
}
add_filter('woocommerce_product_review_list_args', function($args) {
    $args['callback'] = 'custom_review_list';
    return $args;
});




// Admin paneline yorum başlığı kolonunu ekle
function add_comment_title_column($columns) {
    $new_columns = array();

    // Mevcut kolonları döngüye al ve yazar kolonundan sonra başlık kolonunu ekle
    foreach($columns as $key => $value) {
        $new_columns[$key] = $value;
        if($key === 'author') {
            $new_columns['comment_title'] = 'Yorum Başlığı';
        }
    }

    return $new_columns;
}
add_filter('manage_edit-comments_columns', 'add_comment_title_column');

// Kolon içeriğini göster
function show_comment_title_column_content($column, $comment_id) {
    if ($column === 'comment_title') {
        $title = get_comment_meta($comment_id, 'comment_title', true);
        echo esc_html($title ?: '---');
    }
}
add_action('manage_comments_custom_column', 'show_comment_title_column_content', 10, 2);

// Kolonu sıralanabilir yap
function make_comment_title_column_sortable($columns) {
    $columns['comment_title'] = 'comment_title';
    return $columns;
}
add_filter('manage_edit-comments_sortable_columns', 'make_comment_title_column_sortable');

// Sıralama işlemini gerçekleştir
function sort_comment_title_column($query) {
    if (!is_admin()) {
        return;
    }

    $orderby = $query->get('orderby');
    if ('comment_title' === $orderby) {
        $query->set('meta_key', 'comment_title');
        $query->set('orderby', 'meta_value');
    }
}
add_action('pre_get_comments', 'sort_comment_title_column');

// Quick Edit ve Bulk Edit'e başlık alanı ekle
function add_quick_edit_comment_title($column_name, $comment_type) {
    if ('comment_title' === $column_name && 'comment' === $comment_type) {
        ?>
        <fieldset>
            <div class="inline-edit-col">
                <label>
                    <span class="title">Yorum Başlığı</span>
                    <span class="input-text-wrap">
                        <input type="text" name="comment_title" class="comment_title" value="" />
                    </span>
                </label>
            </div>
        </fieldset>
        <?php
    }
}
add_action('quick_edit_custom_box', 'add_quick_edit_comment_title', 10, 2);

// Quick Edit verilerini kaydet
function save_quick_edit_comment_title($comment_id) {
    if (isset($_POST['comment_title'])) {
        update_comment_meta($comment_id, 'comment_title', sanitize_text_field($_POST['comment_title']));
    }
}
add_action('edit_comment', 'save_quick_edit_comment_title');

// Yorum düzenleme sayfasına başlık alanı ekle
function add_comment_meta_box() {
    add_meta_box(
        'comment_title_meta_box',
        'Yorum Başlığı',
        'comment_title_meta_box_content',
        'comment',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes_comment', 'add_comment_meta_box');

// Meta box içeriği
function comment_title_meta_box_content($comment) {
    $comment_title = get_comment_meta($comment->comment_ID, 'comment_title', true);
    wp_nonce_field('comment_title_update', 'comment_title_nonce');
    ?>
    <table class="form-table">
        <tr>
            <td>
                <input type="text" name="comment_title" value="<?php echo esc_attr($comment_title); ?>" style="width: 100%;" />
            </td>
        </tr>
    </table>
    <?php
}

// Yorum düzenleme sayfasında başlık kaydını güncelle
function save_comment_title_meta_box($comment_id) {
    if (!isset($_POST['comment_title_nonce']) || !wp_verify_nonce($_POST['comment_title_nonce'], 'comment_title_update')) {
        return;
    }

    if (isset($_POST['comment_title'])) {
        update_comment_meta($comment_id, 'comment_title', sanitize_text_field($_POST['comment_title']));
    }
}
add_action('edit_comment', 'save_comment_title_meta_box');



// Özel yorumları listele
function custom_woocommerce_product_review_list($comment, $args, $depth) {
    $comment_id = $comment->comment_ID;
    $rating = get_comment_meta($comment_id, 'rating', true);
    $title = get_comment_meta($comment_id, 'title', true);
    $images = get_comment_meta($comment_id, 'images', true);

    ?>
    <li id="comment-<?php echo $comment_id; ?>" class="review">
        <div class="comment_container">
            <div class="comment-text">
                <p class="meta">
                    <strong class="woocommerce-review__author"><?php echo get_comment_author($comment_id); ?></strong>
                    <time class="woocommerce-review__published-date" datetime="<?php echo get_comment_date('c', $comment_id); ?>"><?php echo get_comment_date('', $comment_id); ?></time>
                </p>
                <?php if ($rating): ?>
                    <div class="star-rating" role="img" aria-label="Rated <?php echo $rating; ?> out of 5">
                        <span style="width:<?php echo ($rating / 5) * 100; ?>%">Rated <strong class="rating"><?php echo $rating; ?></strong> out of 5</span>
                    </div>
                <?php endif; ?>
                <?php if ($title): ?>
                    <h4 class="review-title"><?php echo esc_html($title); ?></h4>
                <?php endif; ?>
                <?php comment_text($comment_id); ?>
                <?php if ($images && is_array($images)): ?>
                    <div class="review-images">
                        <?php foreach ($images as $image_url): ?>
                            <img src="<?php echo esc_url($image_url); ?>" alt="Review Image" />
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </li>
    <?php
}

// AJAX ile daha fazla yorum yükleme
function load_more_reviews() {
    $product_id = $_POST['product_id'];
    $page = $_POST['page'];
    $comments_per_page = 4;

    $args = array(
        'post_id' => $product_id,
        'status' => 'approve',
        'orderby' => 'comment_date',
        'order' => 'DESC',
        'number' => $comments_per_page,
        'offset' => ($page - 1) * $comments_per_page,
    );

    $comments = get_comments($args);

    ob_start();
    wp_list_comments(array(
        'callback' => 'custom_woocommerce_product_review_list',
        'style' => 'ol',
        'short_ping' => true,
    ), $comments);
    $html = ob_get_clean();

    $has_more = count($comments) == $comments_per_page;

    wp_send_json(array(
        'html' => $html,
        'has_more' => $has_more,
    ));
}
add_action('wp_ajax_load_more_reviews', 'load_more_reviews');
add_action('wp_ajax_nopriv_load_more_reviews', 'load_more_reviews');