<?php
if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

class Blocks_List_Table extends WP_List_Table {
    
    public function __construct() {
        parent::__construct([
            'singular' => 'block',
            'plural'   => 'blocks',
            'ajax'     => false
        ]);
    }

    public function get_columns() {
        return [
            'cb'           => '<input type="checkbox" />',
            'name'         => 'Block Name',
            'shortcode'    => 'Block Short Code',
            'block_type'   => 'Block Type',
            'created_info' => 'Created',
            'updated_info' => 'Updated'
        ];
    }

    public function get_sortable_columns() {
        return [
            'name'         => ['name', true],
            'block_type'   => ['block_type', false],
            'created_info' => ['created_at', false],
            'updated_info' => ['updated_at', false]
        ];
    }

    protected function get_table_classes() {
        return ['widefat', 'fixed', 'striped', $this->_args['plural']];
    }

    public function prepare_items() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'blocks';
        
        // Process bulk and single actions
        $this->process_bulk_action();
        
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = array($columns, $hidden, $sortable);
        
        $per_page = 10;
        $current_page = $this->get_pagenum();
        
        // Build WHERE clause
        $where_clauses = array();
        
        // Search query
        $search_term = isset($_REQUEST['s']) ? trim($_REQUEST['s']) : '';
        if (!empty($search_term)) {
            $where_clauses[] = $wpdb->prepare(
                "name LIKE %s OR title LIKE %s OR description LIKE %s",
                '%' . $wpdb->esc_like($search_term) . '%',
                '%' . $wpdb->esc_like($search_term) . '%',
                '%' . $wpdb->esc_like($search_term) . '%'
            );
        }
        
        // Block type filter
        $block_type = isset($_REQUEST['block_type']) ? sanitize_text_field($_REQUEST['block_type']) : '';
        if (!empty($block_type)) {
            $where_clauses[] = $wpdb->prepare("block_type = %s", $block_type);
        }
        
        // Combine WHERE clauses
        $where = '';
        if (!empty($where_clauses)) {
            $where = " WHERE " . implode(' AND ', $where_clauses);
        }

        $orderby = isset($_REQUEST['orderby']) ? sanitize_sql_orderby($_REQUEST['orderby']) : 'name';
        $order = isset($_REQUEST['order']) ? sanitize_text_field($_REQUEST['order']) : 'ASC';

        // Get total items
        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name" . $where);

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);

        $offset = ($current_page - 1) * $per_page;
        
        $this->items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name" . $where . " ORDER BY $orderby $order LIMIT %d OFFSET %d",
            $per_page,
            $offset
        ));
    }

    public function get_bulk_actions() {
        return [
            'delete' => 'Delete'
        ];
    }

    public function process_bulk_action() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'blocks';

        // Single delete
        if ('delete' === $this->current_action()) {
            $block_id = isset($_GET['block']) ? absint($_GET['block']) : 0;
            
            if ($block_id) {
                check_admin_referer('delete_block_' . $block_id);
                
                $wpdb->delete(
                    $table_name,
                    ['id' => $block_id],
                    ['%d']
                );
                
                wp_redirect(admin_url('admin.php?page=dn-blocks&message=deleted'));
                exit;
            }
        }
        
        // Bulk delete
        if ('delete' === $this->current_action() && isset($_POST['block'])) {
            check_admin_referer('bulk-' . $this->_args['plural']);
            
            $delete_ids = array_map('absint', $_POST['block']);
            
            foreach ($delete_ids as $id) {
                $wpdb->delete(
                    $table_name,
                    ['id' => $id],
                    ['%d']
                );
            }
            
            wp_redirect(admin_url('admin.php?page=dn-blocks&message=deleted'));
            exit;
        }
    }

    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="block[]" value="%s" />',
            $item->id
        );
    }

    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'shortcode':
                return sprintf('[dn_block id="%d"]', $item->id);
            case 'block_type':
                return esc_html(block_types($item->block_type));
            case 'created_info':
                $user = get_userdata($item->created_by);
                $username = $user ? esc_html($user->display_name) : 'Unknown';
                $date = mysql2date('j F Y, H:i', $item->created_at);
                return sprintf('<strong>%s</strong><br/>%s', $username, $date);
            case 'updated_info':
                $user = get_userdata($item->updated_by);
                $username = $user ? esc_html($user->display_name) : 'Unknown';
                $date = mysql2date('j F Y, H:i', $item->updated_at);
                return sprintf('<strong>%s</strong><br/>%s', $username, $date);
            default:
                return esc_html($item->{$column_name});
        }
    }

    public function column_name($item) {
        $actions = [
            'edit' => sprintf(
                '<a href="?page=dn-blocks-new&id=%s">Edit</a>',
                $item->id
            ),
            'duplicate' => sprintf(
                '<a href="?page=%s&action=duplicate&block=%s">Duplicate</a>',
                $_REQUEST['page'],
                $item->id
            ),
            'delete' => sprintf(
                '<a href="%s" onclick="return confirm(\'Are you sure?\')">Delete</a>',
                wp_nonce_url(
                    sprintf('?page=%s&action=delete&block=%s', $_REQUEST['page'], $item->id),
                    'delete_block_' . $item->id
                )
            )
        ];

        return sprintf(
            '%1$s %2$s',
            '<strong><a href="?page=dn-blocks-new&id=' . $item->id . '">' . esc_html($item->name) . '</a></strong>',
            $this->row_actions($actions)
        );
    }

    public function no_items() {
        echo 'No blocks found.';
    }

    // Add search box display method
    public function search_box($text, $input_id) {
        if (empty($_REQUEST['s']) && !$this->has_items()) {
            return;
        }
        
        $input_id = $input_id . '-search-input';
        
        if (!empty($_REQUEST['orderby'])) {
            echo '<input type="hidden" name="orderby" value="' . esc_attr($_REQUEST['orderby']) . '" />';
        }
        if (!empty($_REQUEST['order'])) {
            echo '<input type="hidden" name="order" value="' . esc_attr($_REQUEST['order']) . '" />';
        }
        if (!empty($_REQUEST['page'])) {
            echo '<input type="hidden" name="page" value="' . esc_attr($_REQUEST['page']) . '" />';
        }
        ?>
        <p class="search-box">
            <label class="screen-reader-text" for="<?php echo esc_attr($input_id); ?>"><?php echo $text; ?>:</label>
            <input type="search" id="<?php echo esc_attr($input_id); ?>" name="s" value="<?php echo esc_attr(isset($_REQUEST['s']) ? $_REQUEST['s'] : ''); ?>" />
            <?php submit_button($text, '', '', false, array('id' => 'search-submit')); ?>
        </p>
        <?php
    }

    // Add extra tablenav method for filters
    public function extra_tablenav($which) {
        if ($which !== 'top') {
            return;
        }
        
        $block_type = isset($_REQUEST['block_type']) ? sanitize_text_field($_REQUEST['block_type']) : '';
        $block_types = block_types();
        ?>
        <div class="alignleft actions">
            <select name="block_type">
                <option value="">All Block Types</option>
                <?php foreach ($block_types as $key => $label) : ?>
                    <option value="<?php echo esc_attr($key); ?>" <?php selected($block_type, $key); ?>>
                        <?php echo esc_html($label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <?php submit_button('Filter', '', 'filter_action', false); ?>
        </div>
        <?php
    }
} 