<?php

global $wpdb;

if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
}

function leanfit_theme_language_files() {
    load_theme_textdomain( 'leanfit', get_template_directory() . '/languages' );
}
add_action( 'after_setup_theme', 'leanfit_theme_language_files' );

// CSS & JS

function leanfittheme_scripts() {
    wp_enqueue_style('leanfit-theme-style', get_stylesheet_uri());
    wp_enqueue_style('leanfit-theme-main', get_template_directory_uri() . '/assets/catalog/view/theme/leanfit/custom/web.css');
    wp_enqueue_style('leanfit-theme-woocommerce', get_template_directory_uri() . '/assets/css/woocommerce.css');
    wp_enqueue_style('leanfit-theme-app', get_template_directory_uri() . '/assets/css/app.css');


    wp_enqueue_script('leanfit-theme-js', get_template_directory_uri() . '/assets/catalog/view/theme/leanfit/custom/app.js');
    wp_enqueue_script('leanfit-theme-js', get_template_directory_uri() . '/assets/js/dn-block.js', array('jquery'), '1.0', true);
    wp_enqueue_script('scripts', get_template_directory_uri() . '/assets/js/scripts.js');

    // AJAX Sepet İçin
    wp_localize_script('leanfit-theme-js', 'ajax_object', array(
        'ajax_url' => admin_url('admin-ajax.php')
    ));
}
add_action('wp_enqueue_scripts', 'leanfittheme_scripts');

function register_my_menus() {
    register_nav_menus(
        array(
            'header-menu' => __('Header Menu'),
            'footer-menu' => __('Footer Menu'),
            'information-pages' => __('Information Pages')
        )
    );
}
add_action('after_setup_theme', 'register_my_menus');

function get_menu_items_by_location($menu_slug, $mobile = false, $array = false) {
    $menu_locations = get_nav_menu_locations();

    if (!isset($menu_locations[$menu_slug])) {
        return []; // Menü bulunamazsa boş döndür
    }

    $menu = wp_get_nav_menu_object($menu_locations[$menu_slug]);
    if (!$menu) return [];

    $menu_items = wp_get_nav_menu_items($menu->term_id);

    $formatted_items = [];
    $html = "";
    if ($menu_items) {
        foreach ($menu_items as $item) {
            if ($array) {
                $formatted_items[] = [
                    'ID' => $item->ID,
                    'title' => !empty($item->title) ? $item->title : get_the_title($item->object_id),
                    'url' => $item->url,
                    'type' => $item->type,
                ];
            } else {
                $title = !empty($item->title) ? $item->title : get_the_title($item->object_id);
                if ($mobile) {  
                    $html .= "<li><a class='uk-accordion-title' href='".$item->url."'>".$title."</a></li>";
                } else {
                    $html .= "<li><a href='".$item->url."'>".$title."</a></li>";
                }
            }
        }
    }


    if ($array)
        return $formatted_items;
    else
        echo $html;
}

function theme_logo() {
    $custom_logo = get_custom_logo();

    preg_match('/src="([^"]+)"/', $custom_logo, $matches);

    if ( isset($matches[1]) ) {
        $logo_url = $matches[1];
        return $logo_url;
    }
}

function custom_template_redirect() {
    // Sadece anasayfa için özel yönlendirme yap
    if (is_front_page() && !is_home()) {
        include(get_template_directory() . '/index.php');
        exit;
    }

    // Kategori sayfası için yönlendirme
    if (is_tax('product_cat')) {
        include(get_template_directory() . '/taxonomy-product_cat.php');
        exit;
    }
}
add_action('template_redirect', 'custom_template_redirect');

function dn_block_shortcode($atts) {
    global $wpdb;
    $atts = shortcode_atts(array(
        'id' => 0,
    ), $atts, 'dn_block');

    $block_id = intval($atts['id']);
    $table_name = $wpdb->prefix . 'blocks';
    $block = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $block_id));

    $clean_json = stripslashes($block->data);
    $data_array = json_decode($clean_json, true);

    if ($block) {
        $block_type = basename($block->block_type);
        $file_path = plugin_dir_path(__FILE__) . "blocks/{$block_type}.php";

        if (file_exists($file_path)) {
            ob_start();
            include $file_path;
            return ob_get_clean();
        }
    }

    return '';
}
add_shortcode('dn_block', 'dn_block_shortcode');

function register_custom_admin_assets() {
    $screen = get_current_screen();
    
    // CSS dosya yolu
    $css_file = get_template_directory() . '/assets/admin/css/admin-style.css';
    $css_url = get_template_directory_uri() . '/assets/admin/css/admin-style.css';

    // JS dosya yolu
    $js_file = get_template_directory() . '/assets/admin/js/admin-script.js';
    $js_url = get_template_directory_uri() . '/assets/admin/js/admin-script.js';

    // CSS dosyası kontrolü ve yükleme
    if (file_exists($css_file)) {
        wp_register_style(
            'custom-admin-style',
            $css_url,
            array(),
            filemtime($css_file)
        );
        wp_enqueue_style('custom-admin-style');
    }

    // JS dosyası kontrolü ve yükleme
    if (file_exists($js_file)) {
        wp_enqueue_script('jquery-ui-sortable');

        wp_register_script(
            'custom-admin-script',
            $js_url,
            array('jquery', 'jquery-ui-sortable'),
            filemtime($js_file),
            true
        );
        wp_enqueue_script('custom-admin-script');

        wp_localize_script(
            'custom-admin-script',
            'adminAjax',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('custom-admin-nonce')
            )
        );
    }
}
add_action('admin_enqueue_scripts', 'register_custom_admin_assets');

if (file_exists(__DIR__ . '/methods/statics.php')) {
    require_once __DIR__ . '/methods/statics.php';
}

if (file_exists(__DIR__ . '/methods/dumper.php')) {
    require_once __DIR__ . '/methods/dumper.php';
}

if (file_exists(__DIR__ . '/methods/contact.php')) {
    require_once __DIR__ . '/methods/contact.php';
}

if (file_exists(__DIR__ . '/methods/custom-field.php')) {
    require_once __DIR__ . '/methods/custom-field.php';
}

if (file_exists(__DIR__ . '/methods/page.php')) {
    require_once __DIR__ . '/methods/page.php';
}

if (file_exists(__DIR__ . '/methods/comments.php')) {
    require_once __DIR__ . '/methods/comments.php';
}

if (file_exists(__DIR__ . '/methods/cart.php')) {
    require_once __DIR__ . '/methods/cart.php';
}

if (file_exists(__DIR__ . '/methods/admin_product.php')) {
    require_once __DIR__ . '/methods/admin_product.php';
}

// Ürün arama AJAX işleyicisi
add_action('wp_ajax_search_products', 'search_products_ajax');
function search_products_ajax() {
    $search = sanitize_text_field($_GET['search']);
    
    $args = array(
        'post_type' => 'product',
        'posts_per_page' => 10,
        's' => $search,
        'orderby' => 'title',
        'order' => 'ASC'
    );
    
    $query = new WP_Query($args);
    $items = array();
    
    if($query->have_posts()) {
        while($query->have_posts()) {
            $query->the_post();
            $items[] = array(
                'id' => get_the_ID(),
                'text' => get_the_title()
            );
        }
    }
    
    wp_reset_postdata();
    
    wp_send_json(['items' => $items]);
}

// Yorum arama AJAX işleyicisi
add_action('wp_ajax_search_comments', 'search_comments_ajax');
function search_comments_ajax() {
    $search = sanitize_text_field($_GET['search']);
    $product_id = isset($_GET['product_id']) ? (int)$_GET['product_id'] : 0;
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $per_page = 10;
    
    $args = array(
        'post_id' => $product_id,
        'search' => $search,
        'number' => $per_page,
        'offset' => ($page - 1) * $per_page,
        'status' => 'approve'
    );
    
    $comments_query = new WP_Comment_Query;
    $comments = $comments_query->query($args);
    $items = array();
    
    foreach($comments as $comment) {
        $items[] = array(
            'id' => $comment->comment_ID,
            'text' => wp_trim_words($comment->comment_content, 10) . ' - ' . $comment->comment_author
        );
    }
    
    wp_send_json([
        'items' => $items,
        'more' => count($comments) === $per_page
    ]);
}

// WooCommerce hesap sayfaları için özel template
function custom_woocommerce_account_template($template) {
    if (is_account_page()) {
        $template = get_template_directory() . '/woocommerce/myaccount/page-account.php';
    }
    return $template;
}
add_filter('template_include', 'custom_woocommerce_account_template', 50);

// Override WooCommerce lost password confirmation template
function override_lost_password_confirmation_template($template, $template_name, $template_path) {
    if ('myaccount/lost-password-confirmation.php' === $template_name) {
        $template = get_stylesheet_directory() . '/woocommerce/myaccount/lost-password-confirmation.php';
    }
    return $template;
}
add_filter('woocommerce_locate_template', 'override_lost_password_confirmation_template', 10, 3);

// İlleri getir
add_action('wp_ajax_get_cities', 'get_cities_ajax');
add_action('wp_ajax_nopriv_get_cities', 'get_cities_ajax');
function get_cities_ajax() {
    check_ajax_referer('update-order-review', 'security');
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'iller';
    
    $cities = $wpdb->get_results("SELECT id, il_adi as name FROM {$table_name} ORDER BY il_adi ASC", ARRAY_A);
    
    if ($wpdb->last_error) {
        wp_send_json_error(['message' => 'Database error occurred']);
        return;
    }
    
    wp_send_json_success($cities);
}

// İlçeleri getir
add_action('wp_ajax_get_districts', 'get_districts_callback');
add_action('wp_ajax_nopriv_get_districts', 'get_districts_callback');
function get_districts_callback() {
    check_ajax_referer('update-order-review', 'security');
    
    $city_id = isset($_POST['city_id']) ? intval($_POST['city_id']) : 0;
    if (!$city_id) {
        wp_send_json_error(['message' => 'Invalid city ID']);
        return;
    }

    global $wpdb;
    $districts = $wpdb->get_results($wpdb->prepare(
        "SELECT id, ilce_adi as name FROM {$wpdb->prefix}ilceler WHERE il_id = %d ORDER BY ilce_adi ASC",
        $city_id
    ), ARRAY_A);

    wp_send_json_success($districts);
}

// Mahalleleri getir
add_action('wp_ajax_get_neighborhoods', 'get_neighborhoods_callback');
add_action('wp_ajax_nopriv_get_neighborhoods', 'get_neighborhoods_callback');
function get_neighborhoods_callback() {
    check_ajax_referer('update-order-review', 'security');
    
    $district_id = isset($_POST['district_id']) ? intval($_POST['district_id']) : 0;
    if (!$district_id) {
        wp_send_json_error(['message' => 'Invalid district ID']);
        return;
    }

    global $wpdb;
    $neighborhoods = $wpdb->get_results($wpdb->prepare(
        "SELECT id, semt_adi as name FROM {$wpdb->prefix}semtler WHERE ilce_id = %d ORDER BY semt_adi ASC",
        $district_id
    ), ARRAY_A);

    wp_send_json_success($neighborhoods);
}

// Adres kaydet
function save_address_callback() {
    check_ajax_referer('woocommerce-process_checkout', 'nonce');
    
    if (!is_user_logged_in()) {
        wp_send_json_error(array('message' => __('You must be logged in to save addresses.', 'woocommerce')));
        return;
    }

    parse_str($_POST['form_data'], $form_data);
    
    $user_id = get_current_user_id();
    $address_type = sanitize_text_field($form_data['address_type']);
    
    global $wpdb;
    $data = array(
        'user_id' => $user_id,
        'address_type' => $address_type,
        'address_title' => sanitize_text_field($form_data['address_title']),
        'first_name' => sanitize_text_field($form_data['first_name']),
        'last_name' => sanitize_text_field($form_data['last_name']),
        'phone' => sanitize_text_field($form_data['phone']),
        'city_id' => intval($form_data['city']),
        'district_id' => intval($form_data['district']),
        'neighborhood_id' => intval($form_data['neighborhood']),
        'address_1' => sanitize_textarea_field($form_data['address_1'])
    );

    $format = array(
        '%d', // user_id
        '%s', // address_type
        '%s', // address_title
        '%s', // first_name
        '%s', // last_name
        '%s', // phone
        '%d', // city_id
        '%d', // district_id
        '%d', // neighborhood_id
        '%s'  // address_1
    );

    if (isset($form_data['address_id']) && !empty($form_data['address_id'])) {
        $wpdb->update(
            $wpdb->prefix . 'user_addresses',
            $data,
            array('id' => intval($form_data['address_id'])),
            $format,
            array('%d')
        );
    } else {
        $wpdb->insert(
            $wpdb->prefix . 'user_addresses',
            $data,
            $format
        );
    }

    if ($wpdb->last_error) {
        wp_send_json_error(array('message' => __('Error saving address.', 'woocommerce')));
        return;
    }

    wp_send_json_success();
}
add_action('wp_ajax_save_address', 'save_address_callback');

// Varsayılan adres ayarla
function set_default_address_callback() {
    check_ajax_referer('woocommerce-process_checkout', 'nonce');
    
    if (!is_user_logged_in()) {
        wp_send_json_error();
        return;
    }

    $address_id = isset($_POST['address_id']) ? intval($_POST['address_id']) : 0;
    $address_type = isset($_POST['address_type']) ? sanitize_text_field($_POST['address_type']) : '';
    
    if (!$address_id || !in_array($address_type, array('billing', 'shipping'))) {
        wp_send_json_error();
        return;
    }

    $user_id = get_current_user_id();
    
    global $wpdb;
    $address = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}user_addresses 
        WHERE id = %d AND user_id = %d AND address_type = %s",
        $address_id,
        $user_id,
        $address_type
    ));

    if (!$address) {
        wp_send_json_error();
        return;
    }

    // Adresi WooCommerce'e kaydet
    $address_fields = array(
        'first_name' => $address->first_name,
        'last_name' => $address->last_name,
        'address_1' => $address->address_1,
        'city' => $wpdb->get_var($wpdb->prepare(
            "SELECT il_adi FROM {$wpdb->prefix}iller WHERE id = %d",
            $address->city_id
        )),
        'district' => $wpdb->get_var($wpdb->prepare(
            "SELECT ilce_adi FROM {$wpdb->prefix}ilceler WHERE id = %d",
            $address->district_id
        )),
        'country' => 'TR'
    );

    foreach ($address_fields as $key => $value) {
        update_user_meta($user_id, $address_type . '_' . $key, $value);
    }

    wp_send_json_success();
}
add_action('wp_ajax_set_default_address', 'set_default_address_callback');

function create_custom_addresses_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'user_addresses';
    
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        address_type varchar(20) NOT NULL,
        address_title varchar(100) NOT NULL,
        first_name varchar(100) NOT NULL,
        last_name varchar(100) NOT NULL,
        phone varchar(20) NOT NULL,
        company varchar(100),
        country varchar(50) DEFAULT 'TR',
        city_id int(11) NOT NULL,
        district_id int(11) NOT NULL,
        neighborhood_id int(11) NOT NULL,
        address_1 text NOT NULL,
        is_selected tinyint(1) DEFAULT 0,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY user_id (user_id),
        KEY address_type (address_type)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
}
add_action('init', 'create_custom_addresses_table');

// Adres seçildiğinde diğer adreslerin seçimini kaldır
function update_selected_address($user_id, $address_id, $address_type) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'user_addresses';

    // Önce aynı tipteki tüm adreslerin seçimini kaldır
    $wpdb->update(
        $table_name,
        array('is_selected' => 0),
        array(
            'user_id' => $user_id,
            'address_type' => $address_type
        ),
        array('%d'),
        array('%d', '%s')
    );

    // Seçilen adresi güncelle
    $wpdb->update(
        $table_name,
        array('is_selected' => 1),
        array(
            'id' => $address_id,
            'user_id' => $user_id,
            'address_type' => $address_type
        ),
        array('%d'),
        array('%d', '%d', '%s')
    );
}

// Yeni adres eklendiğinde ilk adres ise otomatik seç
function maybe_select_first_address($user_id, $address_id, $address_type) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'user_addresses';

    // Bu tipteki adres sayısını kontrol et
    $count = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM $table_name WHERE user_id = %d AND address_type = %s",
        $user_id,
        $address_type
    ));

    // Eğer bu ilk adres ise otomatik olarak seç
    if ($count === 1) {
        update_selected_address($user_id, $address_id, $address_type);
    }
}

// Seçili adres silindiğinde başka bir adresi seç
function handle_selected_address_deletion($user_id, $address_type) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'user_addresses';

    // Bu tipteki ilk adresi bul ve seç
    $first_address = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM $table_name WHERE user_id = %d AND address_type = %s LIMIT 1",
        $user_id,
        $address_type
    ));

    if ($first_address) {
        update_selected_address($user_id, $first_address->id, $address_type);
    }
}

// Adres düzenleme işleyicisi
add_action('wp_ajax_edit_address', 'handle_edit_address');
function handle_edit_address() {
    try {
        // Nonce kontrolü
        if (!isset($_POST['woocommerce-edit-address-nonce']) || !wp_verify_nonce($_POST['woocommerce-edit-address-nonce'], 'woocommerce-edit_address')) {
            throw new Exception(__('Security check failed.', 'woocommerce'));
        }

        // Kullanıcı kontrolü
        $user_id = get_current_user_id();
        if (!$user_id) {
            throw new Exception(__('You must be logged in to edit addresses.', 'woocommerce'));
        }

        // Gerekli alanları kontrol et
        $required_fields = ['address_id', 'address_type', 'address_title', 'first_name', 'last_name', 'city', 'district', 'neighborhood', 'address_1'];
        foreach ($required_fields as $field) {
            if (!isset($_POST[$field]) || empty($_POST[$field])) {
                throw new Exception(sprintf(__('%s is required.', 'woocommerce'), str_replace('_', ' ', $field)));
            }
        }

        $address_id = intval($_POST['address_id']);
        $address_type = sanitize_text_field($_POST['address_type']);

        // Adres tipini kontrol et
        if (!in_array($address_type, ['billing', 'shipping'])) {
            throw new Exception(__('Invalid address type.', 'woocommerce'));
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'user_addresses';

        // Adresin kullanıcıya ait olduğunu kontrol et
        $existing_address = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d AND user_id = %d",
            $address_id,
            $user_id
        ));

        if (!$existing_address) {
            throw new Exception(__('Address not found or access denied.', 'woocommerce'));
        }

        // Veritabanı güncellemesi için veri hazırla
        $data = array(
            'address_title' => sanitize_text_field($_POST['address_title']),
            'first_name' => sanitize_text_field($_POST['first_name']),
            'last_name' => sanitize_text_field($_POST['last_name']),
            'company' => isset($_POST['company']) ? sanitize_text_field($_POST['company']) : '',
            'city_id' => intval($_POST['city']),
            'district_id' => intval($_POST['district']),
            'neighborhood_id' => intval($_POST['neighborhood']),
            'address_1' => sanitize_text_field($_POST['address_1']),
            'updated_at' => current_time('mysql')
        );

        $where = array(
            'id' => $address_id,
            'user_id' => $user_id
        );

        $format = array(
            '%s', // address_title
            '%s', // first_name
            '%s', // last_name
            '%s', // company
            '%d', // city_id
            '%d', // district_id
            '%d', // neighborhood_id
            '%s', // address_1
            '%s'  // updated_at
        );

        $where_format = array(
            '%d', // id
            '%d'  // user_id
        );

        // Veritabanını güncelle
        $result = $wpdb->update($table_name, $data, $where, $format, $where_format);

        if ($result === false) {
            throw new Exception(__('Failed to update address in database.', 'woocommerce'));
        }

        // Başarılı yanıt döndür
        wp_send_json_success(array(
            'message' => __('Address updated successfully.', 'woocommerce'),
            'address' => array_merge($data, array('id' => $address_id))
        ));

    } catch (Exception $e) {
        wp_send_json_error(array(
            'message' => $e->getMessage()
        ));
    }
}

// Yeni adres ekleme AJAX handler
function handle_add_address() {
    check_ajax_referer('woocommerce-add_address', 'woocommerce-add-address-nonce');
    
    global $wpdb;
    $user_id = get_current_user_id();
    
    if (!$user_id) {
        wp_send_json_error(['message' => __('User not logged in', 'woocommerce')]);
        return;
    }

    $address_data = [
        'user_id' => $user_id,
        'address_title' => sanitize_text_field($_POST['address_title']),
        'first_name' => sanitize_text_field($_POST['first_name']),
        'last_name' => sanitize_text_field($_POST['last_name']),
        'company' => !empty($_POST['company']) ? sanitize_text_field($_POST['company']) : '',
        'country' => sanitize_text_field($_POST['country']),
        'city_id' => intval($_POST['city']),
        'district_id' => intval($_POST['district']),
        'neighborhood_id' => intval($_POST['neighborhood']),
        'address_1' => sanitize_text_field($_POST['address_1']),
        'created_at' => current_time('mysql'),
        'updated_at' => current_time('mysql')
    ];

    $success = true;
    $error_message = '';
    $table_name = $wpdb->prefix . 'user_addresses';

    // Fatura ve teslimat adresi olarak iki kayıt oluştur
    foreach (['billing', 'shipping'] as $type) {
        $address_data['address_type'] = $type;
        
        $result = $wpdb->insert(
            $table_name,
            $address_data,
            [
                '%d', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s', '%s', '%s'
            ]
        );

        if ($result) {
            $new_address_id = $wpdb->insert_id;
            maybe_select_first_address($user_id, $new_address_id, $type);
        } else {
            $success = false;
            $error_message = $wpdb->last_error;
            break;
        }
    }

    if ($success) {
        wp_send_json_success(['message' => __('Address added successfully', 'woocommerce')]);
    } else {
        wp_send_json_error([
            'message' => __('Error adding address', 'woocommerce'),
            'technical_error' => $error_message
        ]);
    }
}

// Adres silme işlemi için AJAX handler
function handle_delete_address() {
    check_ajax_referer('delete-address', 'security');

    if (!is_user_logged_in()) {
        wp_send_json_error(['message' => __('You must be logged in to delete addresses.', 'woocommerce')]);
        return;
    }

    $address_id = isset($_POST['address_type']) ? intval($_POST['address_type']) : 0;
    $user_id = get_current_user_id();

    if (!$address_id) {
        wp_send_json_error(['message' => __('Invalid address ID.', 'woocommerce')]);
        return;
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'user_addresses';

    // Adresin kullanıcıya ait olduğunu ve tipini kontrol et
    $address = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_name WHERE id = %d AND user_id = %d",
        $address_id,
        $user_id
    ));

    if (!$address) {
        wp_send_json_error(['message' => __('Address not found or access denied.', 'woocommerce')]);
        return;
    }

    // Adresi sil
    $deleted = $wpdb->delete(
        $table_name,
        ['id' => $address_id, 'user_id' => $user_id],
        ['%d', '%d']
    );

    if ($deleted) {
        // Eğer silinen adres seçili ise, başka bir adresi seç
        if ($address->is_selected) {
            handle_selected_address_deletion($user_id, $address->address_type);
        }
        wp_send_json_success(['message' => __('Address deleted successfully.', 'woocommerce')]);
    } else {
        wp_send_json_error(['message' => __('Failed to delete address.', 'woocommerce')]);
    }
}

// Add custom endpoint for reviews
add_action('init', 'add_reviews_endpoint');
function add_reviews_endpoint() {
    add_rewrite_endpoint('reviews', EP_ROOT | EP_PAGES);
}

// Add reviews tab to my account menu
add_filter('woocommerce_account_menu_items', 'add_reviews_tab', 40);
function add_reviews_tab($menu_items) {
    $menu_items['reviews'] = __('My Reviews', 'woocommerce');
    return $menu_items;
}

// Add content handler for reviews endpoint
add_action('woocommerce_account_reviews_endpoint', 'my_account_reviews_endpoint_content');
function my_account_reviews_endpoint_content() {
    wc_get_template('myaccount/reviews.php');
}

// Sepet ürün miktarı güncelleme
add_action('wp_ajax_update_cart_item_quantity', 'update_cart_item_quantity_callback');
add_action('wp_ajax_nopriv_update_cart_item_quantity', 'update_cart_item_quantity_callback');

function update_cart_item_quantity_callback() {
    check_ajax_referer('update-order-review', 'security');

    $cart_item_key = isset($_POST['cart_item_key']) ? sanitize_text_field($_POST['cart_item_key']) : '';
    $quantity = isset($_POST['quantity']) ? (int)$_POST['quantity'] : 0;

    if (!$cart_item_key || $quantity < 1) {
        wp_send_json_error(['message' => __('Invalid quantity or cart item', 'woocommerce')]);
        return;
    }

    // Sepet öğesini kontrol et
    $cart_item = WC()->cart->get_cart_item($cart_item_key);
    if (!$cart_item) {
        wp_send_json_error(['message' => __('Cart item not found', 'woocommerce')]);
        return;
    }

    // Stok kontrolü
    $product = wc_get_product($cart_item['product_id']);
    if ($product && !$product->has_enough_stock($quantity)) {
        wp_send_json_error(['message' => __('Not enough stock', 'woocommerce')]);
        return;
    }

    // Miktarı güncelle
    $passed_validation = apply_filters('woocommerce_update_cart_validation', true, $cart_item_key, $cart_item, $quantity);
    
    if ($passed_validation) {
        WC()->cart->set_quantity($cart_item_key, $quantity, true);
        
        // Sepeti yeniden hesapla
        WC()->cart->calculate_totals();

        // Güncel sepet bilgilerini döndür
        $cart_totals = [
            'cart_hash' => WC()->cart->get_cart_hash(),
            'cart_total' => WC()->cart->get_cart_total(),
            'cart_subtotal' => WC()->cart->get_cart_subtotal(),
            'items_count' => WC()->cart->get_cart_contents_count()
        ];

        wp_send_json_success($cart_totals);
    } else {
        wp_send_json_error(['message' => __('Cart update failed', 'woocommerce')]);
    }
}

// Seçili adresi güncelleme için AJAX handler
add_action('wp_ajax_update_selected_address', 'handle_update_selected_address');
add_action('wp_ajax_nopriv_update_selected_address', 'handle_update_selected_address');

function handle_update_selected_address() {
    try {
        // Nonce kontrolü
        if (!isset($_POST['security']) || !wp_verify_nonce($_POST['security'], 'update-order-review')) {
            throw new Exception(__('Security check failed.', 'woocommerce'));
        }

        if (!is_user_logged_in()) {
            throw new Exception(__('You must be logged in to update addresses.', 'woocommerce'));
        }

        $address_id = isset($_POST['address_id']) ? intval($_POST['address_id']) : 0;
        $address_type = isset($_POST['address_type']) ? sanitize_text_field($_POST['address_type']) : '';
        $user_id = get_current_user_id();

        if (!$address_id || !in_array($address_type, ['billing', 'shipping'])) {
            throw new Exception(__('Invalid address data.', 'woocommerce'));
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'user_addresses';

        // Adresin kullanıcıya ait olduğunu kontrol et
        $address = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d AND user_id = %d AND address_type = %s",
            $address_id,
            $user_id,
            $address_type
        ));

        if (!$address) {
            throw new Exception(__('Address not found.', 'woocommerce'));
        }

        // Önce tüm adreslerin seçimini kaldır
        $wpdb->update(
            $table_name,
            ['is_selected' => 0],
            [
                'user_id' => $user_id,
                'address_type' => $address_type
            ],
            ['%d'],
            ['%d', '%s']
        );

        // Seçilen adresi güncelle
        $result = $wpdb->update(
            $table_name,
            ['is_selected' => 1],
            ['id' => $address_id],
            ['%d'],
            ['%d']
        );

        if ($result === false) {
            throw new Exception(__('Failed to update address selection.', 'woocommerce'));
        }

        // WooCommerce adres bilgilerini güncelle
        $address_fields = [
            'first_name' => $address->first_name,
            'last_name' => $address->last_name,
            'company' => $address->company,
            'phone' => $address->phone,
            'address_1' => $address->address_1,
            'city' => $wpdb->get_var($wpdb->prepare(
                "SELECT il_adi FROM {$wpdb->prefix}iller WHERE id = %d",
                $address->city_id
            )),
            'district' => $wpdb->get_var($wpdb->prepare(
                "SELECT ilce_adi FROM {$wpdb->prefix}ilceler WHERE id = %d",
                $address->district_id
            )),
            'postcode' => '',
            'country' => 'TR'
        ];

        foreach ($address_fields as $key => $value) {
            update_user_meta($user_id, $address_type . '_' . $key, $value);
        }

        wp_send_json_success([
            'message' => __('Address updated successfully.', 'woocommerce'),
            'address' => $address_fields
        ]);

    } catch (Exception $e) {
        wp_send_json_error([
            'message' => $e->getMessage()
        ]);
    }
}

function enqueue_woocommerce_address_scripts() {
    if (is_account_page()) {
        wp_enqueue_script('woocommerce');
        wp_localize_script('woocommerce', 'woocommerce_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'wc_ajax_url' => WC_AJAX::get_endpoint("%%endpoint%%")
        ));
    }
}
add_action('wp_enqueue_scripts', 'enqueue_woocommerce_address_scripts');

// Kapıda ödeme açıklamasını değiştir
add_filter('woocommerce_gateway_description', 'custom_cod_description', 10, 2);
function custom_cod_description($description, $payment_id) {
    if ($payment_id === 'cod') {
        return ''; // Açıklamayı boş bırak veya istediğiniz metni buraya yazın
    }
    return $description;
}

// Terms modal styles and scripts
function add_terms_modal_styles() {
    if (is_checkout()) {
        ?>
        <style>
            #terms-modal .uk-modal-dialog {
                background: #fff;
            }
            #terms-modal .uk-padding-large {
                padding: 40px;
                max-width: 1200px;
                margin: 0 auto;
            }
            #terms-modal h2 {
                font-size: 24px;
                color: #333;
                margin-bottom: 20px;
            }
            @media (max-width: 960px) {
                #terms-modal .uk-padding-large {
                    padding: 20px;
                }
            }
        </style>
        <script>
            jQuery(document).ready(function($) {
                // Checkbox'ı modal kapandığında otomatik işaretle
                UIkit.util.on('#terms-modal', 'hide', function() {
                    $('#terms').prop('checked', true);
                });
            });
        </script>
        <?php
    }
}
add_action('wp_footer', 'add_terms_modal_styles');

function set_my_locale($locale) {
//    if ( !is_admin() ) {
//        return 'en_US';
//    }
    return 'tr_TR';
}
add_filter('locale', 'set_my_locale');

// Custom order cancellation functionality
add_action('wp_ajax_cancel_order', 'custom_cancel_order');
function custom_cancel_order() {
    if (!isset($_POST['order_id']) || !wp_verify_nonce($_POST['nonce'], 'cancel_order_nonce')) {
        wp_send_json_error('Invalid request');
        return;
    }

    $order_id = intval($_POST['order_id']);
    $order = wc_get_order($order_id);

    if (!$order) {
        wp_send_json_error('Order not found');
        return;
    }

    // Check if current user can cancel this order
    if (!current_user_can('cancel_order', $order_id) && $order->get_customer_id() !== get_current_user_id()) {
        wp_send_json_error('Permission denied');
        return;
    }

    // Check if order status is cancellable
    if (!in_array($order->get_status(), array('pending', 'processing'))) {
        wp_send_json_error('This order cannot be cancelled. Please contact us for assistance.');
        return;
    }

    // Cancel the order
    $order->update_status('cancelled', __('Order cancelled by customer.', 'woocommerce'));
    
    wp_send_json_success(array(
        'message' => __('Order has been cancelled successfully.', 'woocommerce'),
        'redirect' => wc_get_account_endpoint_url('orders')
    ));
}

// Add nonce to footer for AJAX requests
add_action('wp_footer', 'add_cancel_order_nonce');
function add_cancel_order_nonce() {
    if (is_account_page()) {
        echo '<script>var cancelOrderNonce = "' . wp_create_nonce('cancel_order_nonce') . '";</script>';
    }
}

// Sipariş Takip AJAX işleyicisi
add_action('wp_ajax_track_order', 'handle_track_order');
add_action('wp_ajax_nopriv_track_order', 'handle_track_order');

function handle_track_order() {
    // Nonce kontrolü
    if (!wp_verify_nonce($_POST['nonce'], 'track_order_nonce')) {
        wp_send_json_error(__('Güvenlik doğrulaması başarısız.', 'leanfit'));
    }

    $order_id = sanitize_text_field($_POST['order_id']);
    
    // Sipariş numarası kontrolü
    if (empty($order_id)) {
        wp_send_json_error(__('Lütfen sipariş numarası giriniz.', 'leanfit'));
    }

    // Siparişi getir
    $order = wc_get_order($order_id);
    
    if (!$order) {
        wp_send_json_error(__('Sipariş bulunamadı.', 'leanfit'));
    }

    // Sipariş ürünlerini hazırla
    $order_items = array();
    foreach ($order->get_items() as $item_id => $item) {
        $product = $item->get_product();
        if (!$product) continue;

        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $discount_percentage = 0;

        if ($regular_price && $sale_price) {
            $discount_percentage = round((($regular_price - $sale_price) / $regular_price) * 100);
        }

        $order_items[] = array(
            'name' => $item->get_name(),
            'quantity' => $item->get_quantity(),
            'total' => $item->get_total(),
            'image' => wp_get_attachment_image_url($product->get_image_id(), 'woocommerce_thumbnail'),
            'regular_price' => $regular_price * $item->get_quantity(),
            'sale_price' => $item->get_total(),
            'discount_percentage' => $discount_percentage
        );
    }

    // Sipariş durumuna göre ilerleme
    $progress = 0;
    switch ($order->get_status()) {
        case 'pending':
            $progress = 25;
            break;
        case 'processing':
            $progress = 50;
            break;
        case 'shipped':
        case 'on-hold':
            $progress = 75;
            break;
        case 'completed':
            $progress = 100;
            break;
    }

    // Tahmini teslimat tarihi
    $delivery_start = strtotime('+7 days', $order->get_date_created()->getTimestamp());
    $delivery_end = strtotime('+14 days', $order->get_date_created()->getTimestamp());
    $estimated_delivery = date_i18n('F d', $delivery_start) . '-' . date_i18n('F d, Y', $delivery_end);

    // Sipariş bilgilerini hazırla
    $order_data = array(
        'order_number' => $order->get_order_number(),
        'status' => $order->get_status(),
        'status_text' => wc_get_order_status_name($order->get_status()),
        'date' => $order->get_date_created()->format('d/m/Y'),
        'total_items' => count($order_items) . ' ' . _n('Ürün', 'Ürünler', count($order_items), 'leanfit'),
        'payment_method' => $order->get_payment_method_title(),
        'total' => $order->get_formatted_order_total(),
        'shipping_method' => $order->get_shipping_method(),
        'estimated_delivery' => $estimated_delivery,
        'tracking_number' => $order->get_meta('_tracking_number'),
        'progress' => $progress,
        'items' => $order_items,
        'dates' => array(
            'created' => $order->get_date_created() ? $order->get_date_created()->format('d/m/Y') : '',
            'processing' => $order->get_date_modified() && $order->get_status() !== 'pending' ? $order->get_date_modified()->format('d/m/Y') : '',
            'shipped' => $order->get_meta('_date_shipped') ? date('d/m/Y', strtotime($order->get_meta('_date_shipped'))) : '',
            'completed' => $order->get_date_completed() ? $order->get_date_completed()->format('d/m/Y') : ''
        )
    );

    wp_send_json_success($order_data);
}

// Sipariş takip sayfası için endpoint ve sayfa oluştur
function add_track_order_endpoint() {
    add_rewrite_endpoint('track-order', EP_ROOT | EP_PAGES);
    
    // Eğer sayfa yoksa oluştur
    $page_exists = get_page_by_path('track-order');
    if (!$page_exists) {
        $page_data = array(
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_author' => 1,
            'post_name' => 'track-order',
            'post_title' => __('Sipariş Takip', 'leanfit'),
            'post_content' => '',
            'comment_status' => 'closed',
            'page_template' => 'page-track-order.php'
        );
        wp_insert_post($page_data);
    }
}
add_action('init', 'add_track_order_endpoint');

// Sipariş takip sayfası URL'sini al
function get_track_order_url() {
    $page = get_page_by_path('track-order');
    if ($page) {
        return get_permalink($page->ID);
    }
    return home_url('/track-order/');
}

// AJAX handler for loading cart modal content
function load_cart_modal_content() {
    ob_start();
    include(get_template_directory() . '/components/cart-modal.php');
    $cart_modal_content = ob_get_clean();
    echo $cart_modal_content;
    wp_die();
}
add_action('wp_ajax_load_cart_modal', 'load_cart_modal_content');
add_action('wp_ajax_nopriv_load_cart_modal', 'load_cart_modal_content');

// AJAX handler for removing cart items
function remove_cart_item_handler() {
    $cart_item_key = $_POST['cart_item_key'];
    
    if ($cart_item_key) {
        WC()->cart->remove_cart_item($cart_item_key);
        wp_send_json_success();
    } else {
        wp_send_json_error();
    }
    
    wp_die();
}
add_action('wp_ajax_remove_cart_item', 'remove_cart_item_handler');
add_action('wp_ajax_nopriv_remove_cart_item', 'remove_cart_item_handler');

// Yorum formunu özelleştir
add_filter('comment_form_fields', 'custom_comment_form_fields');
function custom_comment_form_fields($fields) {
    if (!is_product()) {
        return $fields;
    }

    $commenter = wp_get_current_commenter();

    $fields['rating'] = '<div class="uk-margin comment-form-rating">
        <label for="rating">' . esc_html__('Değerlendirmeniz', 'leanfit') . '</label>
        <div class="rating-stars">
            <input type="radio" id="star5" name="rating" value="5" />
            <label for="star5" uk-icon="icn-star-2"></label>
            <input type="radio" id="star4" name="rating" value="4" />
            <label for="star4" uk-icon="icn-star-2"></label>
            <input type="radio" id="star3" name="rating" value="3" />
            <label for="star3" uk-icon="icn-star-2"></label>
            <input type="radio" id="star2" name="rating" value="2" />
            <label for="star2" uk-icon="icn-star-2"></label>
            <input type="radio" id="star1" name="rating" value="1" />
            <label for="star1" uk-icon="icn-star-2"></label>
        </div>
    </div>';

    $fields['comment'] = '<div class="uk-margin">
        <textarea id="comment" name="comment" class="uk-textarea" rows="5" placeholder="' . esc_attr__('Yorumunuz', 'leanfit') . '" required></textarea>
    </div>';

    // Photo Reviews for WooCommerce ile entegrasyon
    $fields['photo'] = '<div class="uk-margin">
        <div class="photo-upload-area" id="photo-upload-area">
            <label for="review-photos">' . esc_html__('Fotoğraf Ekle', 'leanfit') . '</label>
            <input type="file" name="review-photos[]" id="review-photos" multiple accept="image/*" class="uk-hidden" />
            <div class="upload-instructions">' . esc_html__('Fotoğraf yüklemek için tıklayın veya sürükleyin', 'leanfit') . '</div>
            <div class="uploaded-photos uk-grid uk-grid-small uk-child-width-auto" uk-grid></div>
        </div>
    </div>';

    return $fields;
}

// Yorum kaydedilmeden önce kontrol
add_filter('preprocess_comment', 'validate_product_review');
function validate_product_review($commentdata) {
    if (!is_product()) {
        return $commentdata;
    }

    if (!isset($_POST['rating']) || empty($_POST['rating'])) {
        wp_die(esc_html__('Lütfen bir değerlendirme puanı seçin', 'leanfit'));
    }

    return $commentdata;
}

// Yorum kaydedilirken rating'i kaydet
add_action('comment_post', 'save_product_review_rating');
function save_product_review_rating($comment_id) {
    if (isset($_POST['rating']) && is_product()) {
        add_comment_meta($comment_id, 'rating', intval($_POST['rating']));
    }
}

// Yorum onaylanmadan önce kontrol
add_filter('pre_comment_approved', 'custom_pre_comment_approved', 10, 2);
function custom_pre_comment_approved($approved, $commentdata) {
    if (is_product()) {
        return 0; // Yorumları onay beklemede tut
    }
    return $approved;
}

// Kullanıcı kaydı sırasında meta verileri kaydet
function save_registration_user_meta($user_id) {
    if (isset($_POST['billing_first_name'])) {
        $name_parts = explode(' ', sanitize_text_field($_POST['billing_first_name']));
        $first_name = array_shift($name_parts);
        $last_name = implode(' ', $name_parts);

        update_user_meta($user_id, 'first_name', $first_name);
        update_user_meta($user_id, 'last_name', $last_name);
        update_user_meta($user_id, 'billing_first_name', $first_name);
        update_user_meta($user_id, 'billing_last_name', $last_name);
        update_user_meta($user_id, 'shipping_first_name', $first_name);
        update_user_meta($user_id, 'shipping_last_name', $last_name);
    }
}
add_action('woocommerce_created_customer', 'save_registration_user_meta');

// WooCommerce Photo Reviews CSS ve JS dosyalarını yükle
function add_woo_photo_reviews_assets() {
    if (is_product()) {
        // CSS dosyaları
        wp_enqueue_style('woo-photo-reviews-style', plugins_url('/woocommerce-photo-reviews/css/style.css'), array(), '1.0.0');
        wp_enqueue_style('woo-photo-reviews-masonry', plugins_url('/woocommerce-photo-reviews/css/masonry.css'), array(), '1.0.0');
        wp_enqueue_style('woo-photo-reviews-shortcode', plugins_url('/woocommerce-photo-reviews/css/shortcode-style.css'), array(), '1.0.0');
        wp_enqueue_style('woo-photo-reviews-lightbox', plugins_url('/woocommerce-photo-reviews/css/lightbox.css'), array(), '1.0.0');
        wp_enqueue_style('woo-photo-reviews-swipebox', plugins_url('/woocommerce-photo-reviews/css/swipebox.css'), array(), '1.0.0');
        wp_enqueue_style('woo-photo-reviews-flexslider', plugins_url('/woocommerce-photo-reviews/css/flexslider.css'), array(), '1.0.0');
    }
}
add_action("wp_enqueue_scripts", "add_woo_photo_reviews_assets", 99);

?>